# -*- coding: utf-8 -*-
"""
EP Directory — Dial API (Annotated)
Generated: 2025-10-20 16:38:59

This annotated version adds high-level documentation and inline comments explaining:
- Flask endpoints used by the UI.
- Pexip Edge/MGMT interactions (auth, queries).
- Poly camera control (REST/SSH).
- gunicorn/systemd & Apache reverse-proxy rationale.
- Environment variables and timeouts.

Functionality is unchanged.
"""
#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import re
import io
import json
import time
import socket
import logging
import threading
import shutil
import subprocess
from typing import List, Dict, Any, Optional
from urllib.parse import quote

import requests  # HTTP client used for Pexip & Poly calls

# ---------------------------------------------------------------------------
# Application Architecture (overview)
# ---------------------------------------------------------------------------
# - Browser → Apache (reverse proxy) → Flask (gunicorn) on 127.0.0.1:5005
# - Frontend uses pexrtc.js to join VMR and render video.
# - Backend exposes:
#     /endpoints, /endpoints_demo1, /vmrs
#     /dial (HOST token → dial → drop)
#     /participants, /participants_status
#     /api/poly/* (REST/SSH camera control)
# - No database; state is in-memory only.
# - Environment is provided by /etc/cklab/dial_api.env .
# ---------------------------------------------------------------------------

from flask import Flask, request, jsonify, Response, send_file

# -----------------------------------------------------------------------------
# Config (MGMT for configuration APIs; EDGE for ONLINE/status and Client API)
# -----------------------------------------------------------------------------
MGMT_HOST = os.environ.get("MGMT_HOST", "cklab-pexmgr.ck-collab-engtest.com").strip()
EDGE_HOST = os.environ.get("EDGE_HOST", "cklab-edges.ck-collab-engtest.com").strip()

MGMT_BASE = os.environ.get("PEXIP_BASE", f"https://{MGMT_HOST}").rstrip("/")
EDGE_BASE = os.environ.get("PEXIP_EDGE_BASE", f"https://{EDGE_HOST}").rstrip("/")

PEXIP_USER = os.environ.get("MGMT_USER", os.environ.get("PEXIP_USER", "admin"))
PEXIP_PASS = os.environ.get("MGMT_PASS", os.environ.get("PEXIP_PASS", "password"))

# TLS verify flags (allow self-signed in labs)
_verify_mgmt = os.environ.get("MGMT_VERIFY_TLS", os.environ.get("PEXIP_VERIFY", "true")).lower()
_verify_edge = os.environ.get("EDGE_VERIFY_TLS", _verify_mgmt).lower()
MGMT_VERIFY = _verify_mgmt not in ("0", "false", "no")
EDGE_VERIFY = _verify_edge not in ("0", "false", "no")

def _atoi(s: str, default: int) -> int:
    try:
        return int(s)
    except Exception:
        return default

MGMT_TIMEOUT_S = _atoi(os.environ.get("MGMT_TIMEOUT_S", "20"), 20)
EDGE_TIMEOUT_S = _atoi(os.environ.get("EDGE_TIMEOUT_S", str(MGMT_TIMEOUT_S)), MGMT_TIMEOUT_S)

# Dial settings
DIAL_VMR_ALIAS = os.environ.get("DIAL_VMR_ALIAS", "drdemo").strip()
HOST_PIN = os.environ.get("HOST_PIN", "").strip()
DIAL_DEFAULT_DOMAIN = os.environ.get("DIAL_DEFAULT_DOMAIN", "ck-collab-engtest.com").strip()
DIAL_DISPLAY_NAME = os.environ.get("DIAL_DISPLAY_NAME", "EPDir Dialer").strip()
DIAL_PROTOCOL = os.environ.get("DIAL_PROTOCOL", "auto").strip().lower()  # "auto" recommended

# HTTP client timeout used for Client API calls
HTTP_TIMEOUT_S = EDGE_TIMEOUT_S

logging.basicConfig(level=logging.INFO)
log = logging.getLogger("dial_api")

app = Flask(__name__)

# -----------------------------------------------------------------------------
# Helper JSON wrappers
# -----------------------------------------------------------------------------
def json_ok(data: Any, status: int = 200) -> Response:
    return jsonify(data), status

def json_err(msg: str, code: int = 500) -> Response:
    return jsonify({"error": msg}), code

# -----------------------------------------------------------------------------
# HTTP helpers
# -----------------------------------------------------------------------------
def _abs_url(base: str, path_or_url: Optional[str]) -> Optional[str]:
    if not path_or_url:
        return None
    if path_or_url.startswith(("http://", "https://")):
        return path_or_url
    return f"{base}/{path_or_url.lstrip('/')}"

def _req(url: str, *, verify: bool, timeout_s: int) -> requests.Response:
    return requests.get(url, auth=(PEXIP_USER, PEXIP_PASS), timeout=timeout_s, verify=verify)

def pexip_paginated(base_url: str, *, verify: bool, timeout_s: int, base_for_rel: str) -> List[Any]:
    """Return concatenated 'results/objects/items' across pages (list of ANY)."""
    all_items: List[Any] = []
    url = base_url
    while url:
        r = _req(url, verify=verify, timeout_s=timeout_s)
        if r.status_code != 200:
            raise requests.HTTPError(f"PEXIP {r.status_code}: {url}")
        payload = r.json() or {}
        page = payload.get("results") or payload.get("objects") or payload.get("items") or []
        if isinstance(page, list):
            all_items.extend(page)
        url = _abs_url(base_for_rel, payload.get("next") or payload.get("next_page"))
    return all_items

# Robust alias extraction (handles dicts and stringified dicts)
_ALIAS_RE = re.compile(r"""['"]alias['"]\s*:\s*['"]([^'"]+)['"]""")

def _extract_alias_from_row(row: Any) -> Optional[str]:
    """Row may be dict or a stringified dict. Return alias if found."""
    if isinstance(row, dict):
        if row.get("alias"):
            return str(row["alias"]).strip()
        for k in ("endpoint_alias", "display_name", "name"):
            if row.get(k):
                return str(row[k]).strip()
        return None
    if isinstance(row, str):
        m = _ALIAS_RE.search(row)
        if m:
            return m.group(1).strip()
        if "{" not in row and "}" not in row and " " not in row:
            return row.strip()
    return None

def _dedupe_sorted(strings: List[str]) -> List[str]:
    seen = set()
    out: List[str] = []
    for s in strings:
        k = (s or "").strip()
        if not k:
            continue
        lk = k.lower()
        if lk in seen:
            continue
        seen.add(lk)
        out.append(k)
    out.sort()
    return out

def _norm_alias(s: str) -> str:
    s = (s or "").strip()
    if not s:
        return s
    if s.startswith(("sip:", "h323:", "mssip:", "tel:", "rtmp:")) or "@" in s:
        return s
    return f"{s}@{DIAL_DEFAULT_DOMAIN}"

def _node_base() -> str:
    return f"{EDGE_BASE}/api/client/v2/conferences/{DIAL_VMR_ALIAS}"

# -----------------------------------------------------------------------------
# Client API helpers (request token, dial, disconnect host)
# -----------------------------------------------------------------------------
def _request_host_token():
    url = f"{_node_base()}/request_token"
    headers = {"Content-Type": "application/json"}
    if HOST_PIN:
        headers["pin"] = HOST_PIN
    body = {"display_name": DIAL_DISPLAY_NAME}
    r = requests.post(url, json=body, headers=headers, timeout=HTTP_TIMEOUT_S, verify=EDGE_VERIFY)
    try:
        j = r.json()
    except Exception:
        j = {"raw": r.text}
    if r.status_code != 200:
        raise RuntimeError(f"request_token failed ({r.status_code}): {j}")
    res = j.get("result") or {}
    token = res.get("token")
    role = res.get("role")
    if not token:
        raise RuntimeError(f"request_token succeeded but returned no token: {j}")
    return token, role, j

def _dial_with_token(token: str, destination: str):
    url = f"{_node_base()}/dial"
    headers = {"Content-Type": "application/json", "token": token}

    def _post(proto: str):
        body = {
            "role": "GUEST",
            "destination": destination,
            "protocol": proto,  # "sip" | "h323" | "auto"
            "source_display_name": DIAL_DISPLAY_NAME,
        }
        return requests.post(url, json=body, headers=headers, timeout=HTTP_TIMEOUT_S, verify=EDGE_VERIFY)

    resp = _post(DIAL_PROTOCOL)
    if resp.status_code == 400:
        try:
            jj = resp.json()
            if "unsupported protocol" in str(jj).lower() and DIAL_PROTOCOL != "auto":
                resp = _post("auto")
        except Exception:
            pass

    try:
        j = resp.json()
    except Exception:
        j = {"raw": resp.text}
    if resp.status_code != 200:
        raise RuntimeError(f"dial failed ({resp.status_code}): {j}")
    return j

def _disconnect_host(token: str, host_uuid: Optional[str]):
    """Disconnect the Host/API participant created by request_token."""
    if not host_uuid:
        return
    url = f"{_node_base()}/participants/{host_uuid}/disconnect"
    headers = {"Content-Type": "application/json", "token": token}
    try:
        r = requests.post(url, headers=headers, timeout=HTTP_TIMEOUT_S, verify=EDGE_VERIFY)
        log.info("Disconnected host leg %s: %s %s", host_uuid, r.status_code, r.text[:160])
    except Exception:
        log.exception("Failed to disconnect host leg %s", host_uuid)

# -----------------------------------------------------------------------------
# Diagnostics
# -----------------------------------------------------------------------------
@app.route("/", methods=["GET"])
def root():
    return json_ok({
        "app": "cklab dial_api",
        "routes": sorted([str(r) for r in app.url_map.iter_rules()]),
        "mgmt_base": MGMT_BASE,
        "edge_base": EDGE_BASE,
        "mgmt_verify_tls": MGMT_VERIFY,
        "edge_verify_tls": EDGE_VERIFY,
        "mgmt_timeout_s": MGMT_TIMEOUT_S,
        "edge_timeout_s": EDGE_TIMEOUT_S,
    })

@app.route("/_health", methods=["GET"])
def _health():
    return "ok", 200

@app.route("/_routes", methods=["GET"])
def _routes():
    return json_ok(sorted([str(r) for r in app.url_map.iter_rules()]))

@app.route("/api/_echo", methods=["GET", "POST"])
def api_echo():
    return jsonify({"ok": True, "method": request.method, "path": request.path})

# -----------------------------------------------------------------------------
# API: Dial endpoint — place dial then immediately leave (drop Host/API leg)
# -----------------------------------------------------------------------------
@app.route("/dial", methods=["POST"])
def dial():
    try:
        payload = request.get_json(force=True, silent=True) or {}
        name = (payload.get("name") or "").strip()
        if not name:
            return json_err("Missing 'name'", 400)
        destination = _norm_alias(name)

        token, role, token_json = _request_host_token()
        if role != "HOST":
            return json_err(f"Token is not HOST (role={role}). Provide HOST_PIN or adjust policy.", 403)
        host_uuid = (token_json.get("result") or {}).get("participant_uuid")

        dial_resp = _dial_with_token(token, destination)
        _disconnect_host(token, host_uuid)

        return json_ok({
            "status": "dial_sent",
            "conference_alias": DIAL_VMR_ALIAS,
            "remote_alias": destination,
            "host_uuid": host_uuid,
            "client_api_response": dial_resp
        })
    except Exception as e:
        log.exception("dial() failed")
        return json_err(str(e), 500)

# -----------------------------------------------------------------------------
# API: ONLINE devices (registration status)
# -----------------------------------------------------------------------------
REG_ALIAS_PATH = "/api/admin/status/v1/registration_alias/"

def _fetch_registration_aliases(base: str, verify: bool, timeout_s: int) -> Optional[List[str]]:
    url = f"{base}{REG_ALIAS_PATH}"
    r = _req(url, verify=verify, timeout_s=timeout_s)
    if r.status_code == 404:
        return None
    if r.status_code != 200:
        log.warning("Status %s from %s", r.status_code, url)
        return None

    payload = r.json() or {}
    rows = payload.get("results") or payload.get("objects") or payload.get("items") or payload

    aliases: List[str] = []
    if isinstance(rows, list):
        for row in rows:
            a = _extract_alias_from_row(row)
            if a:
                aliases.append(a)
    elif isinstance(rows, dict):
        for k in rows.keys():
            if isinstance(k, str) and k:
                aliases.append(k.strip())

    return _dedupe_sorted(aliases)

@app.route("/endpoints", methods=["GET"])
def endpoints_online():
    names = _fetch_registration_aliases(EDGE_BASE, EDGE_VERIFY, EDGE_TIMEOUT_S)
    if names is None:
        log.info("registration_alias not found on EDGE; trying MGMT.")
        names = _fetch_registration_aliases(MGMT_BASE, MGMT_VERIFY, MGMT_TIMEOUT_S)
    if names is None:
        log.warning("registration_alias not found on EDGE or MGMT; returning empty ONLINE list.")
        names = []
    return json_ok({"all": names})

# -----------------------------------------------------------------------------
# API: OFFLINE devices (configured devices) -> MGMT
# -----------------------------------------------------------------------------
@app.route("/endpoints_demo1", methods=["GET"])
def endpoints_offline():
    try:
        url = f"{MGMT_BASE}/api/admin/configuration/v1/device/"
        items = pexip_paginated(url, verify=MGMT_VERIFY, timeout_s=MGMT_TIMEOUT_S, base_for_rel=MGMT_BASE)
        names: List[str] = []
        for obj in items:
            a = _extract_alias_from_row(obj)
            if a:
                names.append(a)
        return json_ok({"all": _dedupe_sorted(names)})
    except Exception as e:
        log.exception("/endpoints_demo1 failed (MGMT=%s)", MGMT_BASE)
        return json_err(str(e), 500)

# -----------------------------------------------------------------------------
# API: Virtual Meeting Rooms (conferences) -> MGMT
# -----------------------------------------------------------------------------
@app.route("/vmrs", methods=["GET"])
def vmrs():
    try:
        url = f"{MGMT_BASE}/api/admin/configuration/v1/conference/"
        items = pexip_paginated(url, verify=MGMT_VERIFY, timeout_s=MGMT_TIMEOUT_S, base_for_rel=MGMT_BASE)
        aliases: List[str] = []
        for obj in items:
            a = _extract_alias_from_row(obj)
            if a:
                aliases.append(a)
            if isinstance(obj, dict) and isinstance(obj.get("aliases"), list):
                for inner in obj["aliases"]:
                    ai = _extract_alias_from_row(inner)
                    if ai:
                        aliases.append(ai)
        return json_ok({"all": _dedupe_sorted(aliases)})
    except Exception as e:
        log.exception("/vmrs failed (MGMT=%s)", MGMT_BASE)
        return json_err(str(e), 500)

# -----------------------------------------------------------------------------
# API: participants via Client API (requires being in the room)
# -----------------------------------------------------------------------------
@app.route("/participants", methods=["GET"])
def participants():
    try:
        token, role, _ = _request_host_token()
        if role != "HOST":
            return json_err(f"Token is not HOST (role={role}). Provide HOST_PIN or adjust policy.", 403)
        url = f"{_node_base()}/participants"
        headers = {"token": token}
        r = requests.get(url, headers=headers, timeout=HTTP_TIMEOUT_S, verify=EDGE_VERIFY)
        try:
            j = r.json()
        except Exception:
            j = {"raw": r.text}
        if r.status_code != 200:
            return json_err("Failed to get participants", r.status_code)
        items = j.get("result") if isinstance(j, dict) else j
        results = []
        for p in (items or []):
            results.append({
                "uuid": p.get("uuid") or p.get("participant_uuid") or p.get("id"),
                "display_name": p.get("display_name") or p.get("remote_display_name") or p.get("name") or p.get("alias"),
                "role": p.get("role"),
                "protocol": p.get("protocol")
            })
        return json_ok(results)
    except Exception as e:
        log.exception("participants failed")
        return json_err(str(e), 500)

# -----------------------------------------------------------------------------
# API: disconnect one or more participants via Client API (requires HOST)
# -----------------------------------------------------------------------------
@app.route("/disconnect", methods=["POST"])
def disconnect():
    try:
        payload = request.get_json(force=True, silent=True) or {}
        uuids = payload.get("uuids") or []
        if not isinstance(uuids, list) or not uuids:
            return json_err("Missing 'uuids' array", 400)
        token, role, _ = _request_host_token()
        if role != "HOST":
            return json_err(f"Token is not HOST (role={role}). Provide HOST_PIN or adjust policy.", 403)
        headers = {"token": token, "Content-Type": "application/json"}
        results = []
        for u in uuids:
            u = (u or "").strip()
            if not u:
                continue
            url = f"{_node_base()}/participants/{u}/disconnect"
            r = requests.post(url, headers=headers, timeout=HTTP_TIMEOUT_S, verify=EDGE_VERIFY)
            try:
                j = r.json()
            except Exception:
                j = {"raw": r.text}
            results.append({"uuid": u, "status": r.status_code, "response": j})
        all_ok = all(r["status"] == 200 for r in results)
        return json_ok({"disconnected": results}, 200 if all_ok else 207)
    except Exception as e:
        log.exception("disconnect failed")
        return json_err(str(e), 500)

# -----------------------------------------------------------------------------
# API: status participants (admin/status) — does NOT join the room
# -----------------------------------------------------------------------------
def _fetch_participants_status() -> Optional[list]:
    """
    Query admin/status participant list from EDGE first, then MGMT.
    Returns a list of participant dicts, or None if endpoint missing.
    """
    path = "/api/admin/status/v1/participant/"
    bases = [(EDGE_BASE, EDGE_VERIFY, EDGE_TIMEOUT_S), (MGMT_BASE, MGMT_VERIFY, MGMT_TIMEOUT_S)]
    for base, verify, timeout in bases:
        url = f"{base}{path}"
        try:
            r = _req(url, verify=verify, timeout_s=timeout)
        except Exception as ex:
            log.warning("status fetch error %s: %s", url, ex)
            continue
        if r.status_code == 404:
            continue
        if r.status_code != 200:
            log.warning("status %s from %s", r.status_code, url)
            continue
        data = r.json() or {}
        rows = data.get("results") or data.get("objects") or data.get("items") or data
        if isinstance(rows, list):
            return rows
        if isinstance(rows, dict):
            return list(rows.values())
    return None

@app.route("/participants_status", methods=["GET"])
def participants_status():
    try:
        vmr = request.args.get("alias", DIAL_VMR_ALIAS).strip() or DIAL_VMR_ALIAS
        token, role, _ = _request_host_token()
        if role != "HOST":
            return json_err(f"Token is not HOST (role={role}). Provide HOST_PIN or adjust policy.", 403)

        url = f"{EDGE_BASE}/api/client/v2/conferences/{vmr}/participants"
        headers = {"token": token}
        r = requests.get(url, headers=headers, timeout=HTTP_TIMEOUT_S, verify=EDGE_VERIFY)
        try:
            j = r.json()
        except Exception:
            j = {"raw": r.text}

        if r.status_code != 200:
            return json_err(f"Failed to get participants ({r.status_code})", r.status_code)

        items = j.get("result") if isinstance(j, dict) else j
        out = []
        for p in (items or []):
            out.append({
                "uuid": p.get("uuid") or p.get("participant_uuid") or p.get("id"),
                "display_name": p.get("display_name") or p.get("remote_display_name") or p.get("name") or p.get("alias"),
                "alias": p.get("alias"),
                "remote_alias": p.get("remote_alias"),
                "protocol": p.get("protocol"),
                "role": p.get("role"),
            })
        return jsonify({"all": out}), 200
    except Exception as e:
        log.exception("/participants_status failed")
        return json_err(str(e), 500)

# -----------------------------------------------------------------------------
# Poly camera control (REST + persistent SSH via pexpect)
# -----------------------------------------------------------------------------
import urllib3
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# Device config (single device default; override with env if needed)
POLY_HOST_DEFAULT = os.environ.get("POLY_HOST_DEFAULT", "192.168.0.101").strip()
POLY_USER = os.environ.get("POLY_USER", "admin")
POLY_PASS = (os.environ.get("POLY_PASS", "T3$tW3b@c3ss") or "").rstrip()
POLY_SSH_LOCK = threading.Lock()

def _poly_base_urls() -> list[str]:
    """Try HTTPS first, then HTTP as a fallback."""
    return [f"https://{POLY_HOST_DEFAULT}", f"http://{POLY_HOST_DEFAULT}"]

def _extract_session_id(payload: Any, resp: requests.Response) -> Optional[str]:
    try:
        sid = payload["session"]["sessionId"]
        if isinstance(sid, (str, int)) and str(sid).strip():
            return str(sid)
    except Exception:
        pass
    for h in ("X-Session-Id", "Session-Id", "X-Auth-Token", "X-Token"):
        val = resp.headers.get(h)
        if val:
            return val
    try:
        for name, val in resp.cookies.items():
            if "session" in name.lower() and str(val).strip():
                return str(val)
    except Exception:
        pass
    return None

def _poly_open_session_and_base():
    """Open a REST session to the Poly device and return (session, base_url)."""
    last_err: Optional[Exception] = None
    for base in _poly_base_urls():
        try:
            s = requests.Session()
            s.verify = False
            r = s.post(f"{base}/rest/session",
                       json={"user": POLY_USER, "password": POLY_PASS},
                       timeout=10)
            payload = {}
            try:
                payload = r.json() if r.content else {}
            except Exception:
                pass
            if r.status_code >= 400:
                last_err = RuntimeError(f"{base} /rest/session {r.status_code}: {r.text[:240]}")
                continue
            sid = _extract_session_id(payload, r)
            if not sid:
                last_err = RuntimeError(f"{base} /rest/session ok but no session id")
                continue
            s.cookies.set("sessionID", str(sid))
            s.cookies.set("sessionId", str(sid))
            s.headers["X-Session-Id"] = str(sid)
            return s, base
        except Exception as e:
            last_err = e
    raise last_err or RuntimeError("Poly session failed")

# ---- Cameras list (REST) ----
@app.route("/api/poly/cameras", methods=["GET", "POST"])
def api_poly_cameras():
    try:
        s, base = _poly_open_session_and_base()
        r = s.get(f"{base}/rest/cameras/near/all", timeout=10)
        if r.status_code >= 400:
            return jsonify({"error": f"GET cameras {r.status_code}: {r.text[:240]}"}), 502
        cams = r.json() if r.content else []
        out = []
        arr = cams if isinstance(cams, list) else (cams.get("cameras") or [])
        for c in arr:
            if isinstance(c, dict) and "cameraIndex" in c:
                name = (c.get("name") or "").strip()
                if name:
                    out.append({
                        "cameraIndex": c["cameraIndex"],
                        "name": name,
                        "selected": bool(c.get("selected") or c.get("active") or c.get("current"))
                    })
        return jsonify(out), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

# ---- Presets (REST): store/activate/image ----
@app.route("/api/poly/presets", methods=["GET", "POST"])
def api_poly_presets():
    """List near presets with proxied thumbnails."""
    try:
        s, base = _poly_open_session_and_base()
        r = s.get(f"{base}/rest/cameras/near/presets/all", timeout=10)
        if r.status_code >= 400:
            return jsonify({"error": f"GET presets/all {r.status_code}: {r.text[:240]}"}), 502
        data = r.json() if r.content else []
        rows = data if isinstance(data, list) else data.get("presets", [])
        out = []
        for row in rows:
            if not isinstance(row, dict):
                continue
            idx = row.get("index")
            img = (row.get("imageLocation") or "").strip()
            stored = bool(row.get("stored"))
            if idx is None or not img:
                continue
            proxied = f"/api/poly/preset_image?path={quote(img, safe='')}"
            out.append({"index": int(idx), "stored": stored, "image": proxied})
        return jsonify(out), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@app.route("/api/poly/preset_image", methods=["GET"])
def api_poly_preset_image():
    """Proxy preset thumbnail from the device."""
    path = (request.args.get("path") or "").strip()
    if not path.startswith("/"):
        path = "/" + path
    try:
        s, base = _poly_open_session_and_base()
        r = s.get(f"{base}{path}", timeout=10, stream=True)
        if r.status_code >= 400:
            return jsonify({"error": f"image fetch {r.status_code}"}), 502
        blob = io.BytesIO(r.content)
        return send_file(blob, mimetype=r.headers.get("Content-Type", "image/png"),
                         as_attachment=False, download_name=path.rsplit("/",1)[-1])
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@app.route("/api/poly/preset/activate", methods=["POST"])
def api_poly_preset_activate():
    """Activate near preset by index."""
    try:
        body = request.get_json(force=True, silent=True) or {}
        idx = body.get("preset_index")
        if idx is None:
            return jsonify({"error": "preset_index is required"}), 400
        s, base = _poly_open_session_and_base()
        r = s.post(f"{base}/rest/cameras/near/presets/{int(idx)}",
                   json={"action": "activate"}, timeout=10)
        if r.status_code >= 400:
            return jsonify({"error": f"activate {r.status_code}: {r.text[:240]}"}), 502
        try:
            payload = r.json()
        except Exception:
            payload = {"status": "ok"}
        return jsonify({"message": f"Activated preset {idx}", "result": payload}), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@app.route("/api/poly/preset/store", methods=["POST"])
def api_poly_preset_store():
    """Store a near preset mapped to a specific camera index (pure REST)."""
    try:
        body = request.get_json(force=True, silent=True) or {}
        pidx = body.get("preset_index")
        cidx = body.get("camera_index")
        pname = (body.get("name") or "").strip()
        if pidx is None or cidx is None:
            return jsonify({"error": "preset_index and camera_index are required"}), 400

        s, base = _poly_open_session_and_base()
        payload = {"action": "store", "cameraIndex": int(cidx)}
        if pname:
            payload["name"] = pname

        r = s.post(f"{base}/rest/cameras/near/presets/{int(pidx)}",
                   json=payload, timeout=15)
        if r.status_code >= 400:
            text = r.text[:240] if r.text else f"HTTP {r.status_code}"
            return jsonify({"error": f"store {r.status_code}: {text}"}), 502

        try:
            j = r.json()
        except Exception:
            j = {"status": "ok"}

        return jsonify({
            "message": f"Preset {int(pidx)} stored to near camera index {int(cidx)}.",
            "result": j
        }), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

# ---- Persistent SSH session using pexpect ----
#   - Accepts host key automatically
#   - Waits for password prompt
#   - After login, idles and runs commands on demand
#   - Uses BEGIN/END markers to capture verification blocks
import pexpect

class PolySSHSession:
    def __init__(self, host: str, user: str, pwd: str):
        self.host = host
        self.user = user
        self.pwd = pwd
        self.child: Optional[pexpect.spawn] = None
        self.lock = threading.Lock()
        # Tunables
        self.banner_timeout = int(os.environ.get("POLY_BANNER_TIMEOUT_S", "45"))
        self.cmd_timeout    = int(os.environ.get("POLY_CMD_TIMEOUT_S", "20"))

    def _spawn(self):
        argv = [
            "ssh",
            "-tt",
            "-o", "StrictHostKeyChecking=no",
            "-o", "UserKnownHostsFile=/dev/null",
            "-o", "PreferredAuthentications=password,keyboard-interactive",
            "-o", "PubkeyAuthentication=no",
            "-o", "ConnectTimeout=8",
            "-o", "ServerAliveInterval=10",
            "-o", "ServerAliveCountMax=2",
            f"{self.user}@{self.host}"
        ]
        log.info("[PolySSH] spawning: %s", " ".join(argv))
        self.child = pexpect.spawn(argv[0], argv[1:], encoding="utf-8", timeout=self.banner_timeout)
        # Handle first prompts
        i = self.child.expect([
            r'(?i)are you sure you want to continue connecting \(yes/no(\s*\[fingerprint\])?\)\?',
            r'(?i)password:',
            r'(?i)permission denied',
            r'.*$',  # banner spill
        ], timeout=self.banner_timeout)
        if i == 0:
            self.child.sendline("yes")
            self.child.expect(r'(?i)password:', timeout=self.banner_timeout)
            self.child.sendline(self.pwd)
        elif i == 1:
            self.child.sendline(self.pwd)
        elif i == 2:
            raise RuntimeError("Permission denied before password entry")
        else:
            # Might have printed banner; send a newline to prompt password if needed
            try:
                self.child.expect(r'(?i)password:', timeout=8)
                self.child.sendline(self.pwd)
            except pexpect.TIMEOUT:
                # maybe we're already in the shell (older firmwares)
                pass

        # Prove we are at a working shell by round-tripping a marker
        self.child.sendline("echo __READY__")
        self.child.expect("__READY__", timeout=self.cmd_timeout)
        log.info("[PolySSH] session ready")

    def ensure(self):
        if self.child is None or not self.child.isalive():
            self._spawn()

    def run_camera_switch(self, near_index: int, settle_ms: int = 900) -> Dict[str, Any]:
        """
        Send camera switch + verify block:
          camera near <idx>
          sleep <settle>
          echo __BEGIN__
          show camera near
          echo __END__
        Return dict with stdout chunk and parsed selected index.
        """
        with self.lock:
            for attempt in (1, 2):  # one transparent retry on EOF/closed
                try:
                    self.ensure()
                    c = self.child
                    # Clear any pending output quickly
                    try:
                        c.read_nonblocking(size=4096, timeout=0.1)
                    except Exception:
                        pass

                    MARK_BEG = "__EPDIR_BEGIN__"
                    MARK_END = "__EPDIR_END__"
                    cmd = f"camera near {int(near_index)}"
                    log.info("[PolySSH] sending: %s", cmd)
                    c.sendline(cmd)

                    # Wait for the command echo to appear (device usually echoes the line)
                    try:
                        c.expect(re.escape(cmd), timeout=self.cmd_timeout)
                    except pexpect.TIMEOUT:
                        # not all builds echo; continue
                        pass

                    # settle + verification block
                    c.sendline(f"sleep {settle_ms/1000.0:.2f}")
                    c.sendline(f"echo {MARK_BEG}")
                    c.sendline("show camera near")
                    c.sendline(f"echo {MARK_END}")

                    # Find our markers and capture in-between
                    c.expect(MARK_BEG, timeout=self.cmd_timeout)
                    c.expect(MARK_END, timeout=self.cmd_timeout)
                    block = c.before  # text between markers
                    selected = self._parse_selected(block)
                    log.info("[PolySSH] verify: selected=%s", selected)

                    return {
                        "ok": (selected is None) or (selected == int(near_index)),
                        "selected": selected,
                        "verify_block": block[-2000:],  # tail for debugging
                    }
                except (pexpect.EOF, pexpect.TIMEOUT) as e:
                    log.warning("[PolySSH] run attempt %s failed: %s", attempt, e)
                    try:
                        if self.child:
                            self.child.close(force=True)
                    except Exception:
                        pass
                    self.child = None
                    if attempt == 2:
                        raise
                except Exception:
                    # fatal unexpected error
                    try:
                        if self.child:
                            self.child.close(force=True)
                    except Exception:
                        pass
                    self.child = None
                    raise

    @staticmethod
    def _parse_selected(text: str) -> Optional[int]:
        """
        Parse the 'show camera near' block for a selected index.
        Accepts variations like:
          Near Camera: 3
          selected: 2
          current=1
        """
        if not text:
            return None
        pat = re.compile(r"(?:\bnear\b|\bselected\b|\bcurrent\b)\s*[:=]?\s*(\d+)\b", re.IGNORECASE)
        for ln in text.splitlines():
            m = pat.search(ln)
            if m:
                try:
                    return int(m.group(1))
                except Exception:
                    pass
        # fallback: single integer in block
        nums = re.findall(r"\b(\d+)\b", text)
        if len(nums) == 1:
            try:
                return int(nums[0])
            except Exception:
                pass
        return None

# Global singleton session to the default device
_POLY_SSH = PolySSHSession(
    host=os.environ.get("POLY_HOST_DEFAULT", POLY_HOST_DEFAULT),
    user=os.environ.get("POLY_USER", POLY_USER),
    pwd=(os.environ.get("POLY_PASS", POLY_PASS) or "").rstrip()
)

# ---- Camera switch over persistent SSH ----
@app.route("/api/poly/switch", methods=["POST"])
def api_poly_switch():
    """
    Body: { "camera_index": <int> }
    Attempts persistent SSH session first (pexpect), then falls back to sshpass/OpenSSH, then Paramiko.
    Returns JSON with debug timeline.
    """
    started = time.time()
    dbg = []
    def mark(event, **kw): dbg.append({"ts": round(time.time() - started, 3), "event": event, **kw})

    try:
        body = request.get_json(force=True, silent=True) or {}
    except Exception as e:
        return jsonify({"error": f"bad_json: {e}"}), 400

    idx = body.get("camera_index")
    if idx is None:
        return jsonify({"error": "Missing camera_index"}), 400

    try:
        target_idx = int(idx)
    except Exception:
        return jsonify({"error": "camera_index must be an integer"}), 400

    host = os.environ.get("POLY_HOST_DEFAULT", POLY_HOST_DEFAULT)
    user = os.environ.get("POLY_USER", POLY_USER)
    pwd  = os.environ.get("POLY_PASS", POLY_PASS)
    SETTLE_MS = int(os.environ.get("POLY_SETTLE_MS", "700"))
    SSH_TIMEOUT_S = int(os.environ.get("POLY_SSH_TIMEOUT_S", "15"))

    mark("input", host=host, idx=target_idx)

    if not POLY_SSH_LOCK.acquire(timeout=5):
        return jsonify({"error": "Another camera command is in progress", "debug": dbg}), 423

    def ok_json(method, result=None, out_text=""):
        return jsonify({
            "message": f"Camera switch requested to index {target_idx}.",
            "method": method,
            "verified": bool((result or {}).get("ok", False)),
            "selected": (result or {}).get("selected"),
            "stdout_tail": (out_text or (result or {}).get("verify_block", ""))[-2000:],
            "debug": dbg
        }), 200

    try:
        # ---------- Fast path: persistent SSH session ----------
        try:
            mark("persistent_try")
            res = _POLY_SSH.run_camera_switch(target_idx, settle_ms=SETTLE_MS)
            mark("persistent_ok", selected=res.get("selected"))
            return ok_json("persistent_pexpect", res)
        except Exception as e:
            mark("persistent_fail", err=str(e))

        # ---------- Fallback: sshpass + OpenSSH ----------
        which_sshpass = shutil.which("sshpass") or ""
        which_ssh = shutil.which("ssh") or ""
        if which_sshpass and which_ssh and not os.environ.get("POLY_FORCE_PARAMIKO"):
            chain = f"camera near tracking off ; camera near {target_idx} ; sleep {SETTLE_MS/1000.0:.2f} ; exit"
            ssh_cmd = [
                which_sshpass, "-p", pwd, which_ssh,
                "-tt",
                "-o", "StrictHostKeyChecking=no",
                "-o", "UserKnownHostsFile=/dev/null",
                "-o", "GSSAPIAuthentication=no",
                "-o", "PreferredAuthentications=password,keyboard-interactive",
                "-o", "PubkeyAuthentication=no",
                "-o", "ConnectTimeout=5",
                "-o", "ServerAliveInterval=5",
                "-o", "ServerAliveCountMax=1",
                f"{user}@{host}", chain
            ]
            mark("sshpass_run")
            try:
                proc = subprocess.run(ssh_cmd, capture_output=True, text=True, timeout=SSH_TIMEOUT_S)
                text = (proc.stdout or "") + (proc.stderr or "")
                if proc.returncode == 0:
                    mark("sshpass_ok")
                    return ok_json("sshpass", {"ok": True}, text)
                else:
                    mark("sshpass_rc", rc=proc.returncode)
            except Exception as e:
                mark("sshpass_exc", err=str(e))

        # ---------- Fallback: Paramiko one-shot ----------
        try:
            import paramiko
            sock = socket.create_connection((host, 22), timeout=5)
            trans = paramiko.Transport(sock)
            trans.start_client(timeout=10)
            trans.auth_password(user, pwd, fallback=False)
            chan = trans.open_session(timeout=8)
            chan.get_pty()
            chan.invoke_shell()
            chan.send(f"camera near {target_idx}\n")
            time.sleep(SETTLE_MS/1000.0)
            chan.send("exit\n")
            text = chan.recv(4096).decode("utf-8", "ignore")
            mark("paramiko_ok")
            return ok_json("paramiko", {"ok": True}, text)
        except Exception as e:
            mark("paramiko_fail", err=str(e))
            raise

    except Exception as e:
        log.exception("api_poly_switch failed")
        return jsonify({"error": f"{type(e).__name__}: {e}", "debug": dbg}), 500
    finally:
        try: POLY_SSH_LOCK.release()
        except Exception: pass


@app.route("/api/poly/diag", methods=["GET"])
def api_poly_diag():
    host = os.environ.get("POLY_HOST_DEFAULT", "192.168.0.101").strip()
    user = os.environ.get("POLY_SSH_USER", os.environ.get("POLY_USER", "admin")).strip()

    out = {
        "host": host,
        "user": user,
        "PATH": os.environ.get("PATH", ""),
        "which": {
            "sshpass": shutil.which("sshpass") or "",
            "ssh": shutil.which("ssh") or "",
        },
        "versions": {}
    }

    # versions (stderr for openssh -V)
    try:
        p = subprocess.run(["ssh", "-V"], capture_output=True, text=True, timeout=4)
        out["versions"]["ssh"] = (p.stderr or p.stdout or "").strip()
    except Exception as e:
        out["versions"]["ssh"] = f"ERR: {e}"

    try:
        p = subprocess.run(["sshpass", "-V"], capture_output=True, text=True, timeout=4)
        out["versions"]["sshpass"] = (p.stderr or p.stdout or "").strip()
    except Exception as e:
        out["versions"]["sshpass"] = f"ERR: {e}"

    # TCP check
    try:
        s = socket.create_connection((host, 22), timeout=5)
        out["tcp_22"] = "open"
        s.close()
    except Exception as e:
        out["tcp_22"] = f"closed: {e}"

    # Paramiko import present?
    try:
        import paramiko  # noqa: F401
        out["paramiko_import"] = "ok"
    except Exception as e:
        out["paramiko_import"] = f"ERR: {e}"

    return jsonify(out), 200

# -----------------------------------------------------------------------------
# Entrypoint
# -----------------------------------------------------------------------------
if __name__ == "__main__":
    # Bind to loopback; your reverse proxy should expose this as needed
    app.run(host="127.0.0.1", port=5001, debug=False)
