/**
 * EP Directory — Frontend Library (Annotated)
 * Generated: 2025-10-20 16:38:59
 *
 * Highlights:
 * - Event handlers and UI actions map to backend routes.
 * - WebRTC join via pexrtc.js prior to dialing far endpoints.
 * - Participant roster tracking -> green button synchronization.
 * - Status banner and error handling approaches.
 * - Poly modals for camera switching & presets.
 */
// ===== library.js (robust per-endpoint watcher; VMRs dial-in only) =====
var rtc = null;
var video = null;
var pin = null;
var bandwidth = null;

// ---- Config ----
const PEX_NODE = "cklab-edges.ck-collab-engtest.com";
const PEX_CONF = "drdemo";
const PEX_BW   = "2048";
const PEX_NAME = "Chris Provider";
const PEX_PIN  = "2024";

// ---- State ----
let vmrActive = false;
let __endedOnce = false; // single-shot disconnect cleanup guard

// Participants (for local PexRTC roster / modal)
let participantMap = new Map(); // uuid -> participant object
let selectedParticipantUuid = null;

// Track which phonebook buttons are currently "connected"
const activeEndpoints = new Map(); // endpointKey -> button element

// Watchers to keep buttons green while remote is present (no dialer leg required)
const pollers = new Map();        // endpointKey -> setInterval handle
const presence = new Map();       // endpointKey -> { lastSeen, missCount, everPresent }

// Debounce roster-based clears so we don't fight the watcher
const rosterMisses = new Map();   // endpointKey -> consecutive-miss count

// Raw phonebook caches
let ONLINE_RAW  = []; // /endpoints
let OFFLINE_RAW = []; // /endpoints_demo1
let VMR_RAW     = []; // /vmrs (aliases)

// ============================================================
// Utilities
// ============================================================
function normalizeName(s) {
  s = String(s || "").trim().toLowerCase();
  s = s.replace(/^sip:/i, "");
  s = s.split("@")[0];
  s = s.replace(/\s+/g, "");
  return s;
}
function normKey(s){ return normalizeName(s); }

// ============================================================
// Status + button helpers
// ============================================================
function setStatus(message, type) {
  if (type === void 0) type = "info";
  const el = document.getElementById("reg_status");
  if (!el) return;
  el.textContent = message;
  el.classList.remove("flashing-red", "green");
  if (type === "dialing") el.classList.add("flashing-red");
  if (type === "ok") el.classList.add("green");
}

function clearStatusIfNoActive() {
  if (activeEndpoints.size === 0) {
    const el = document.getElementById("reg_status");
    if (el) {
      el.textContent = "";
      el.classList.remove("flashing-red", "green");
    }
  }
}

function markButtonDialing(btn) {
  if (!btn) return;
  btn.dataset.originalText = btn.dataset.originalText || btn.textContent;
  btn.textContent = "Dialing Your Clinical Endpoint...";
  btn.disabled = true;
}

function markButtonConnected(btn) {
  if (!btn) return;
  btn.textContent = "Connected to Current Session";
  btn.disabled = false;
  btn.classList.add("green","connected");
}

// -------- key-based button lookup & reset ----------
function getButtonByKey(key){
  key = String(key||"");
  const direct = document.querySelector(`[data-endpoint-key="${key}"]`);
  if (direct) return direct;
  const lis = document.querySelectorAll("#videoWindow .department li, #vmrList li");
  for (let i = 0; i < lis.length; i++) {
    const li = lis[i];
    const first = (li.childNodes && li.childNodes[0]) ? li.childNodes[0].textContent : "";
    const leftText = (first || "").replace("Call", "").trim();
    if (normKey(leftText) === key) {
      return li.querySelector(".callBtn, .dialInBtn");
    }
  }
  return null;
}

function stopParticipantWatcher(nameOrKey){
  const key = normKey(nameOrKey);
  if (pollers.has(key)){
    clearInterval(pollers.get(key));
    pollers.delete(key);
  }
  presence.delete(key);
  rosterMisses.delete(key);
}

function resetCallButtonForKey(key) {
  stopParticipantWatcher(key);
  let btn = activeEndpoints.get(key) || getButtonByKey(key);
  if (btn) {
    if (btn.classList.contains("callBtn")) btn.textContent = "Call";
    if (btn.classList.contains("dialInBtn")) btn.textContent = "Dial-in";
    btn.disabled = false;
    btn.classList.remove("green","connected");
  }
  activeEndpoints.delete(key);
  clearStatusIfNoActive();
}

function resetCallButtonForName(name) { resetCallButtonForKey(normKey(name)); }

function resetAllCallButtons() {
  for (const key of pollers.keys()) clearInterval(pollers.get(key));
  pollers.clear();
  presence.clear();
  rosterMisses.clear();
  const btns = document.querySelectorAll(".callBtn, .dialInBtn");
  for (let i = 0; i < btns.length; i++) {
    const b = btns[i];
    if (b.classList.contains("callBtn")) b.textContent = "Call";
    if (b.classList.contains("dialInBtn")) b.textContent = "Dial-in";
    b.disabled = false;
    b.classList.remove("green","connected");
  }
  activeEndpoints.clear();
  clearStatusIfNoActive();
}

// ============================================================
// Keep-green watcher
// ============================================================
function startParticipantWatcher(targetName, onPresent, onGone){
  const key = normKey(targetName);
  stopParticipantWatcher(key);
  const INTERVAL_MS        = 2000;
  const SUCCESS_MISSES_MAX = 3;
  const LAST_SEEN_GRACE_MS = 15000;
  const NEVER_SEEN_CUTOFF  = 20000;

  presence.set(key, { startTime: Date.now(), lastSeen: 0, missCount: 0, everPresent: false });

  function isPresentInRoster() {
    if (!participantMap || typeof participantMap.forEach !== "function") return false;
    let found = false;
    participantMap.forEach(p => {
      const cand = p && (p.display_name || p.remote_display_name || p.alias || p.name || "");
      if (normKey(cand) === key) found = true;
    });
    return found;
  }

  const handle = setInterval(() => {
    const tr = presence.get(key) || { startTime: Date.now(), lastSeen: 0, missCount: 0, everPresent: false };
    const present = isPresentInRoster();

    if (present) {
      tr.everPresent = true;
      tr.lastSeen = Date.now();
      tr.missCount = 0;
      presence.set(key, tr);

      const b = activeEndpoints.get(key) || getButtonByKey(key);
      if (b){
        b.classList.add("green","connected");
        b.disabled = false;
        b.textContent = "Connected to Current Session";
        activeEndpoints.set(key, b);
      }
      onPresent && onPresent();
      return;
    }

    tr.missCount += 1;
    presence.set(key, tr);
    const now = Date.now();

    if (tr.everPresent && tr.lastSeen > 0) {
      const age = now - tr.lastSeen;
      if (age >= LAST_SEEN_GRACE_MS && tr.missCount >= SUCCESS_MISSES_MAX) {
        stopParticipantWatcher(key);
        resetCallButtonForKey(key);
        onGone && onGone();
      }
      return;
    }

    const neverSeenAge = now - tr.startTime;
    if (!tr.everPresent && neverSeenAge >= NEVER_SEEN_CUTOFF && tr.missCount >= SUCCESS_MISSES_MAX) {
      stopParticipantWatcher(key);
      resetCallButtonForKey(key);
      onGone && onGone();
    }
  }, INTERVAL_MS);

  pollers.set(key, handle);
}

// ============================================================
// Start button UX
// ============================================================
function showInactiveStartButton() {
  const startButton = document.getElementById("startMeetingBtn");
  if (!startButton) return;
  startButton.textContent = "Meeting room is currently inactive, please click to reactivate it";
  startButton.classList.add("inactive-top");
  startButton.style.display = "inline-block";
}

function normalizeStartButton() {
  const startButton = document.getElementById("startMeetingBtn");
  if (!startButton) return;
  startButton.textContent = "Start Your Meeting Room";
  startButton.classList.remove("inactive-top");
  startButton.style.display = "none";
}

// ============================================================
// Meeting launcher
// ============================================================
window.launchMeeting = /** Join VMR via pexrtc.js and prepare event callbacks for media/roster. */
function launchMeeting() {
  if (vmrActive) return;
  normalizeStartButton();
  const deviceSettings = document.getElementById("deviceSettingsContainer");
  if (deviceSettings) deviceSettings.style.display = "block";
  try { start(); } catch (e) { console.warn("start() failed:", e); }
  initialise(PEX_NODE, PEX_CONF, PEX_BW, PEX_NAME, PEX_PIN);
};

// ============================================================
// Ghost-leg prevention
// ============================================================
function ensureParticipantNotPresent(targetName) {
  return new Promise(function (resolve) {
    if (!rtc) return resolve();
    try { if (rtc.requestParticipants) rtc.requestParticipants(); } catch (e) {}
    setTimeout(function () {
      const target = normKey(targetName);
      const matches = [];
      participantMap.forEach(function (p) {
        if (normKey(p.display_name) === target) matches.push(p);
      });
      for (let i = 0; i < matches.length; i++) {
        const p = matches[i];
        try { rtc.disconnectParticipant(p.uuid); } catch (e) { console.warn("Pre-disconnect failed:", e); }
      }
      resolve();
    }, 400);
  });
}

// ============================================================
// Dial flow
// ============================================================
function placeDialRequest(name, btn) {
  const key = normKey(name);
  setStatus("Dialing In Your Clinical Endpoint...", "dialing");
  if (btn) markButtonDialing(btn);

  fetch("/dial", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ name: name })
  })
  .then(function (res) {
    return res.json().catch(function () { return {}; }).then(function (data) {
      if (!res.ok) {
        var msg = (data && (data.error || data.detail || data.message)) || ("HTTP " + res.status);
        throw new Error(msg);
      }

      setStatus("Connected to Current Session", "ok");
      if (btn) {
        btn.setAttribute("data-endpoint-name", name);
        btn.setAttribute("data-endpoint-key", key);
        markButtonConnected(btn);
        activeEndpoints.set(key, btn);
      }

      startParticipantWatcher(
        name,
        function(){
          const b = activeEndpoints.get(key) || getButtonByKey(key);
          if (b){
            b.classList.add('green','connected');
            b.disabled = false;
            b.textContent = "Connected to Current Session";
            activeEndpoints.set(key, b);
          }
        },
        function(){
          resetCallButtonForKey(key);
        }
      );
    });
  })
  .catch(function (err) {
    console.error("Dial error:", err);
    setStatus("Dial failed: " + ((err && err.message) ? err.message : err));
    resetCallButtonForKey(key);
  });
}

window.handleCallButtonClick = async function (btn, endpointName) {
  if (!vmrActive) {
    window.launchMeeting();
    setTimeout(async function () {
      try { await ensureParticipantNotPresent(endpointName); } catch (e) { console.warn(e); }
      placeDialRequest(endpointName, btn);
    }, 1200);
  } else {
    try { await ensureParticipantNotPresent(endpointName); } catch (e) { console.warn(e); }
    placeDialRequest(endpointName, btn);
  }
};

// ============================================================
// Participants + PexRTC lifecycle (for local WebRTC & modal)
// ============================================================
function normalizeParticipant(p) {
  return {
    uuid: p.uuid,
    display_name: p.display_name || p.remote_display_name || p.name || p.alias || p.uuid,
    role: p.role,
    is_chair: p.is_chair || p.role === "chair",
    is_local: p.is_local || p.self || false
  };
}
function buildMapFromRoster(roster) {
  const map = new Map();
  const items = Array.isArray(roster) ? roster
              : (roster && Array.isArray(roster.participants)) ? roster.participants
              : [];
  for (let i = 0; i < items.length; i++) {
    const p = items[i];
    const np = normalizeParticipant(p);
    map.set(np.uuid, Object.assign({}, p, np));
  }
  return map;
}
function rosterNameSet() {
  const s = new Set();
  participantMap.forEach(function (p) {
    const d = p && (p.display_name || p.remote_display_name || p.alias || p.name || "");
    s.add(normKey(d));
  });
  return s;
}
function refreshParticipantModalIfOpen() { const modal = document.getElementById("participantModal"); if (modal && modal.style.display !== "none") renderParticipantList(); }

function onRosterList(roster) {
  participantMap = buildMapFromRoster(roster);
  reconcileActiveButtonsWithRoster();
  refreshParticipantModalIfOpen();
}
function onParticipantCreate(p){
  const np=normalizeParticipant(p);
  participantMap.set(np.uuid, Object.assign({}, p, np));
  reconcileActiveButtonsWithRoster();
  refreshParticipantModalIfOpen();
}
function onParticipantUpdate(p){
  const np=normalizeParticipant(p);
  const prev=participantMap.get(np.uuid)||{};
  participantMap.set(np.uuid, Object.assign({}, prev, p, np));
  reconcileActiveButtonsWithRoster();
  refreshParticipantModalIfOpen();
}
function onParticipantDelete(p){
  const np=normalizeParticipant(p);
  if (np.uuid) participantMap.delete(np.uuid);
  refreshParticipantModalIfOpen();
}

// Debounced roster reconciling
function reconcileActiveButtonsWithRoster(){
  const present = rosterNameSet();
  const keys = Array.from(activeEndpoints.keys());
  for (let i=0;i<keys.length;i++){
    const key = keys[i];
    if (pollers.has(key)) { rosterMisses.delete(key); continue; }
    if (present.has(key)) { rosterMisses.delete(key); continue; }
    const n = (rosterMisses.get(key) || 0) + 1;
    rosterMisses.set(key, n);
    if (n >= 3) {
      rosterMisses.delete(key);
      resetCallButtonForKey(key);
    }
  }
}

function renderParticipantList(){
  const listEl=document.getElementById("participantList"); const noneEl=document.getElementById("noParticipantsMsg"); const confirmBtn=document.getElementById("confirmDisconnectBtn");
  if(!listEl||!noneEl||!confirmBtn) return;
  listEl.innerHTML=""; selectedParticipantUuid=null; confirmBtn.disabled=true;
  const entries=[]; participantMap.forEach(function(p){ const nm=normKey(p.display_name)===normKey(PEX_NAME); if(!p.is_local && !p.is_chair && !nm) entries.push(p); });
  if(entries.length===0){ noneEl.style.display="block"; return; }
  noneEl.style.display="none";
  for(let i=0;i<entries.length;i++){ const p=entries[i]; const li=document.createElement("li"); li.style.display="flex"; li.style.alignItems="center"; li.style.justifyContent="space-between"; li.style.padding="6px 0"; li.style.borderBottom="1px solid #eee";
    const left=document.createElement("div"); left.textContent=p.display_name;
    const choose=document.createElement("input"); choose.type="radio"; choose.name="participantSelect"; choose.value=p.uuid; choose.style.marginLeft="10px"; choose.addEventListener("change", function(){ selectedParticipantUuid=p.uuid; confirmBtn.disabled=!selectedParticipantUuid; });
    li.appendChild(left); li.appendChild(choose); listEl.appendChild(li); }
}
window.openParticipantModal=function(){ const modal=document.getElementById("participantModal"); if(!modal) return; const stage=document.querySelector(".videoStage"); if(stage){ const r=stage.getBoundingClientRect(); modal.style.position="fixed"; modal.style.left=r.left+"px"; modal.style.top=r.top+"px"; modal.style.width=r.width+"px"; modal.style.height=r.height+"px"; modal.style.display="flex"; modal.style.alignItems="center"; modal.style.justifyContent="center"; } else { modal.style.display="flex"; modal.style.alignItems="center"; modal.style.justifyContent="center"; } renderParticipantList(); };
window.closeParticipantModal=function(){ const modal=document.getElementById("participantModal"); if(!modal) return; modal.style.display="none"; selectedParticipantUuid=null; const btn=document.getElementById("confirmDisconnectBtn"); if(btn) btn.disabled=true; };
document.addEventListener("DOMContentLoaded", function(){ const confirmBtn=document.getElementById("confirmDisconnectBtn"); if(confirmBtn) confirmBtn.addEventListener("click", function(){ if(!rtc) return alert("Meeting is not active."); if(!selectedParticipantUuid) return alert("Please select a participant."); try{ rtc.disconnectParticipant(selectedParticipantUuid); closeParticipantModal(); }catch(e){ console.error("Failed to disconnect participant:", e); alert("Failed to disconnect participant: "+((e && e.message)?e.message:e)); } }); const allBtn=document.getElementById("disconnectAllBtn"); if(allBtn) allBtn.addEventListener("click", function(){ try{ endCall(); }catch(e){ console.warn(e);} }); });

// ============================================================
// PexRTC lifecycle
// ============================================================
function finalise(){ try{ if(rtc) rtc.disconnectAll(); }catch(e){} if(video) video.src=""; }

function remoteDisconnect(reason){
  if (__endedOnce) { console.log("Already cleaned up; reason:", reason); return; }
  __endedOnce = true;

  vmrActive=false;
  resetAllCallButtons();
  try{ if(rtc) rtc.disconnectAll(); }catch(e){}
  rtc=null;

  const v=document.getElementById("video");
  if(v){ try{ v.srcObject=null; }catch(e){} v.removeAttribute("src"); v.load(); v.style.display="none"; }

  const controls=document.getElementById("controls");
  if(controls) controls.style.display="none";

  showInactiveStartButton();
  const deviceSettingContainer=document.getElementById("deviceSettingsContainer");
  if(deviceSettingContainer) deviceSettingContainer.style.display="none";

  window.removeEventListener("beforeunload", finalise);
  console.log("Disconnected:", reason);
}

function doneSetup(_videoURL, pin_status){
  console.log("PIN status:", pin_status);
  rtc.connect(pin);
}

function connected(videoURL){
  vmrActive=true;
  __endedOnce = false;
  const v=document.getElementById("video");
  if(v){
    v.poster="";
    v.style.display="block";
    if(typeof MediaStream!=="undefined" && videoURL instanceof MediaStream){ v.srcObject=videoURL; }
    else { v.src=videoURL; }
  }
  const controls=document.getElementById("controls");
  if(controls) controls.style.display="block";
  setStatus("Meeting Room live.","ok");
}

function feccHandler(signal){ console.debug("FECC:", signal); }
function handleApplicationMessage(message){ console.debug("AppMessage:", message); }

function initialise(node, conference, userbw, name, userpin){
  video=document.getElementById("video"); pin=userpin; bandwidth=parseInt(userbw,10);
  rtc=new PexRTC(); rtc.fecc_supported=true;
  try{ const camSel=document.querySelector("select#videoSource"); if(camSel && camSel.value){ rtc.video_source=camSel.value; } }catch(e){ console.warn("No camera selector:", e); }
  window.addEventListener("beforeunload", finalise);
  rtc.onSetup=doneSetup; rtc.onConnect=connected; rtc.onError=remoteDisconnect; rtc.onFECC=feccHandler; rtc.onApplicationMessage=handleApplicationMessage;
  rtc.onRosterList=onRosterList; rtc.onParticipantCreate=onParticipantCreate; rtc.onParticipantUpdate=onParticipantUpdate; rtc.onParticipantDelete=onParticipantDelete;
  rtc.makeCall(node, conference, name, bandwidth);
}

window.endCall=function(){
  try {
    if (rtc) { rtc.disconnect(); }
  } catch(e) { console.warn(e); }
  setTimeout(function(){
    if (!__endedOnce) remoteDisconnect("Forced cleanup (timeout)");
  }, 2000);
};
window.endParticipant=function(){ openParticipantModal(); };

// ============================================================
// PexRTC-based local mute handlers (global for inline onclicks)
// ============================================================
let __audioMuted = false;
let __videoMuted = false;

function __setBtnState(btnId, on, iconOn, iconOff) {
  const a = document.getElementById(btnId);
  if (!a) return;
  a.classList.toggle("active", on);
  const i = a.querySelector("i");
  if (i) i.className = "fa " + (on ? iconOff : iconOn);
}

window.muteAudioStreams = function () {
  if (!window.rtc) { console.warn("rtc not ready"); return; }
  __audioMuted = !__audioMuted;
  try { if (typeof rtc.muteAudio === "function") rtc.muteAudio(__audioMuted); } catch (e) { console.warn(e); }
  __setBtnState("id_muteaudio", __audioMuted, "fa-microphone", "fa-microphone-slash");
  console.log("[PexRTC] local audio", __audioMuted ? "muted" : "unmuted");
};

window.muteVideoStreams = function () {
  if (!window.rtc) { console.warn("rtc not ready"); return; }
  __videoMuted = !__videoMuted;
  try { if (typeof rtc.muteVideo === "function") rtc.muteVideo(__videoMuted); } catch (e) { console.warn(e); }
  __setBtnState("id_mutevideo", __videoMuted, "fa-video", "fa-video-slash");
  console.log("[PexRTC] local video", __videoMuted ? "muted" : "unmuted");
};

// ============================================================
// Rendering helpers (ONLINE/OFFLINE)
// ============================================================
function renderCKLabsLists(names){
  const clinicalUL=document.querySelector("#cklab-clinical ul");
  const officeUL  =document.querySelector("#cklab-office   ul");
  if(!clinicalUL||!officeUL) return;
  clinicalUL.innerHTML=""; officeUL.innerHTML="";
  const sorted=names.slice().sort(function(a,b){ return a.localeCompare(b); });
  for(let i=0;i<sorted.length;i++){
    const name=sorted[i];
    const li=document.createElement("li");
    li.textContent=name+" ";
    const btn=document.createElement("button");
    btn.className="callBtn"; btn.textContent="Call";
    btn.setAttribute("data-endpoint-name", name);
    btn.setAttribute("data-endpoint-key", normKey(name));
    btn.onclick=function(){ window.handleCallButtonClick(btn, name); };
    li.appendChild(btn);
    if(name.toLowerCase().includes("ipad")) clinicalUL.appendChild(li); else officeUL.appendChild(li);
  }
}
function renderDemo1Lists(names){
  const clinicalUL=document.querySelector("#demo1-clinical ul");
  const officeUL  =document.querySelector("#demo1-office   ul");
  if(!clinicalUL||!officeUL) return;
  clinicalUL.innerHTML=""; officeUL.innerHTML="";
  const sorted=names.slice().sort(function(a,b){ return a.localeCompare(b); });
  for(let i=0;i<sorted.length;i++){
    const name=sorted[i];
    const li=document.createElement("li");
    li.textContent=name;
    if(name.toLowerCase().includes("ipad")) clinicalUL.appendChild(li); else officeUL.appendChild(li);
  }
}

// ============================================================
// VMR rendering + search (Dial-in only)
// ============================================================
function renderVMRList(aliases){
  const list=document.getElementById("vmrList");
  if(!list) return;
  list.innerHTML="";
  const sorted=aliases.slice().sort(function(a,b){ return a.localeCompare(b); });
  for(let i=0;i<sorted.length;i++){
    const alias=sorted[i];
    const li=document.createElement("li");
    li.style.display="flex";
    li.style.justifyContent="space-between";
    li.style.alignItems="center";
    li.style.padding="6px 0";
    li.style.borderBottom="1px solid #eee";

    const left=document.createElement("div");
    left.textContent=alias;

    const right=document.createElement("div");
    right.className="vmr-actions";

    const dialBtn=document.createElement("button");
    dialBtn.className="callBtn dialInBtn";
    dialBtn.textContent="Dial-in";
    dialBtn.setAttribute("data-endpoint-name", alias);
    dialBtn.setAttribute("data-endpoint-key", normKey(alias));
    dialBtn.onclick=function(){ window.handleCallButtonClick(dialBtn, alias); };

    right.appendChild(dialBtn);
    li.appendChild(left);
    li.appendChild(right);
    list.appendChild(li);
  }
}

window.filterVMRList=function(query){
  const q=(query||"").toLowerCase();
  const u=document.getElementById("vmrList"); if(!u) return;
  const items=u.querySelectorAll("li");
  for(let i=0;i<items.length;i++){
    const li=items[i];
    const name=(li.childNodes && li.childNodes[0] && li.childNodes[0].textContent) ? li.childNodes[0].textContent.toLowerCase() : "";
    li.style.display = name.indexOf(q)!==-1 ? "flex" : "none";
  }
};

// ============================================================
// Filtering for ONLINE and OFFLINE lists
// ============================================================
window.filterEndpointList = function (facilityId, query) {
  const q = (query || "").toLowerCase();
  const facility = document.getElementById(facilityId);
  if (!facility) return;
  const lis = facility.querySelectorAll("ul li");
  for (let i = 0; i < lis.length; i++) {
    const li = lis[i];
    const text = (li.textContent || "").toLowerCase();
    li.style.display = text.indexOf(q) !== -1 ? "" : "none";
  }
};

// ============================================================
// Cross-list de-dupe (ONLINE wins) & render both sections
// ============================================================
function renderBothFiltered(){
  const onlineSetUnique=new Set(); const onlineList=[];
  for(let i=0;i<ONLINE_RAW.length;i++){ const n=(ONLINE_RAW[i]||"").trim(); if(!n) continue; const key=normKey(n); if(!onlineSetUnique.has(key)){ onlineSetUnique.add(key); onlineList.push(n); } }
  const onlineLower=new Set(); for(let i=0;i<onlineList.length;i++) onlineLower.add(normKey(onlineList[i]));
  const offlineSetUnique=new Set(); const offlineList=[];
  for(let i=0;i<OFFLINE_RAW.length;i++){ const n=(OFFLINE_RAW[i]||"").trim(); if(!n) continue; const key=normKey(n); if(onlineLower.has(key)) continue; if(!offlineSetUnique.has(key)){ offlineSetUnique.add(key); offlineList.push(n); } }
  renderCKLabsLists(onlineList);
  renderDemo1Lists(offlineList);
}

// ============================================================
// Fetchers
// ============================================================
window.fetchPexipEndpoints = async function () {
  try {
    setStatus("Fetching CKLabs (ONLINE) endpoints from Pexip...", "dialing");
    const res = await fetch("/endpoints");
    if (!res.ok) throw new Error("Backend " + res.status);
    const data = await res.json();
    ONLINE_RAW = Array.isArray(data.all) ? data.all : [];
    renderBothFiltered();
    setStatus("CKLabs phonebook updated.", "ok");
  } catch (e) {
    console.error(e);
    setStatus("Failed to fetch CKLabs (ONLINE) endpoints from Pexip.");
    alert("Failed to fetch endpoints: " + ((e && e.message) ? e.message : e));
  }
};
window.fetchPexipEndpointsDemo1 = async function () {
  try {
    setStatus("Fetching CKLabs Offline (OFFLINE) endpoints from Pexip...", "dialing");
    const res = await fetch("/endpoints_demo1");
    if (!res.ok) throw new Error("Backend " + res.status);
    const data = await res.json();
    OFFLINE_RAW = Array.isArray(data.all) ? data.all : [];
    renderBothFiltered();
    setStatus("Offline device list updated.", "ok");
  } catch (e) {
    console.error(e);
    setStatus("Failed to fetch Offline devices.");
    alert("Failed to fetch endpoints: " + ((e && e.message) ? e.message : e));
  }
};
window.fetchVMRs = async function () {
  try {
    setStatus("Fetching Virtual Meeting Rooms...", "dialing");
    const res = await fetch("/vmrs");
    if (!res.ok) throw new Error("Backend " + res.status);
    const data = await res.json();
    VMR_RAW = Array.isArray(data.all) ? data.all : [];
    renderVMRList(VMR_RAW);
    setStatus("Virtual Meeting Rooms updated.", "ok");
  } catch (e) {
    console.error(e);
    setStatus("Failed to fetch Virtual Meeting Rooms.");
    alert("Failed to fetch VMRs: " + ((e && e.message) ? e.message : e));
  }
};

// ============================================================
// Global one-click sync
// ============================================================
window.fetchAllEndpoints = async function () {
  const syncBtn = document.getElementById("syncAllBtn");
  try {
    if (syncBtn) { syncBtn.disabled = true; syncBtn.textContent = "Syncing..."; }
    setStatus("Syncing endpoints from Pexip...", "dialing");

    const [onlineRes, offlineRes, vmrRes] = await Promise.all([
      fetch("/endpoints"),
      fetch("/endpoints_demo1"),
      fetch("/vmrs")
    ]);

    if (!onlineRes.ok)  throw new Error("ONLINE backend " + onlineRes.status);
    if (!offlineRes.ok) throw new Error("OFFLINE backend " + offlineRes.status);
    if (!vmrRes.ok)     throw new Error("VMR backend " + vmrRes.status);

    const [onlineData, offlineData, vmrData] = await Promise.all([
      onlineRes.json(), offlineRes.json(), vmrRes.json()
    ]);

    ONLINE_RAW  = Array.isArray(onlineData.all)  ? onlineData.all  : [];
    OFFLINE_RAW = Array.isArray(offlineData.all) ? offlineData.all : [];
    VMR_RAW     = Array.isArray(vmrData.all)     ? vmrData.all     : [];

    renderBothFiltered();
    renderVMRList(VMR_RAW);

    setStatus("Endpoints synced.", "ok");
  } catch (e) {
    console.error("Global sync error:", e);
    setStatus("Failed to sync endpoints/VMRs from Pexip.");
    alert("Failed to sync: " + ((e && e.message) ? e.message : e));
  } finally {
    if (syncBtn) { syncBtn.disabled = false; syncBtn.textContent = "Sync endpoints from Pexip"; }
  }
};

// ===== Per-endpoint button sync from live roster =====
(function(){
  function stripSip(s){
    var x = String(s||"").trim();
    x = x.replace(/^sip:/i, "");
    x = x.split("@")[0];
    return x;
  }
  function keyName(s){ return stripSip(s).toLowerCase().replace(/\s+/g,""); }

  function participantNameSet(){
    var set = new Set();
    try {
      if (window.participantMap && typeof participantMap.forEach === "function"){
        participantMap.forEach(function(p){
          var disp = p && (p.display_name || p.remote_display_name || p.alias || "");
          set.add(keyName(disp));
        });
      }
    } catch(e){}
    return set;
  }

  function allEndpointButtons(){
    var btns = Array.from(document.querySelectorAll("[data-role='callButton'], .callButton, button.call"));
    if (!btns.length){
      btns = Array.from(document.querySelectorAll("#office-endpoints button, .office-endpoints button, .callBtn, .dialInBtn"));
    }
    return btns;
  }

  function labelForButton(btn){
    var name = btn.getAttribute("data-endpoint-name");
    if (name) return keyName(name);
    try{
      var parent = btn.parentElement;
      if (parent){
        var txt = "";
        for (var i=0;i<parent.childNodes.length;i++){
          var n = parent.childNodes[i];
          if (n === btn) break;
          if (n.nodeType === 3) txt += n.textContent || "";
          if (n.nodeType === 1 && !/button/i.test(n.tagName)) txt += " " + (n.textContent||"");
        }
        if (txt.trim()) return keyName(txt);
      }
    }catch(e){}
    var hint = btn.getAttribute("aria-label") || btn.getAttribute("title") || "";
    return keyName(hint);
  }

  function setBtnState(btn, connected){
    var label = connected ? "Connected to Current Session" : "Call";
    if (btn.dataset) btn.dataset.label = label;
    btn.textContent = label;
    if (btn.dataset) btn.dataset.state = connected ? "connected" : "call";
    if (connected) btn.classList.add("green","connected");
    else btn.classList.remove("green","connected");
  }

  function refreshEndpointButtonsFromRoster(){
    var present = participantNameSet();
    var btns = allEndpointButtons();
    btns.forEach(function(btn){
      var explicitKey = btn.getAttribute("data-endpoint-key");
      var nm = explicitKey ? explicitKey : labelForButton(btn);
      if (!nm) return;
      var isConnected = present.has(nm);
      setBtnState(btn, isConnected);
      if (isConnected) activeEndpoints.set(nm, btn);
      else activeEndpoints.delete(nm);
    });
  }

  function bindRtc(){
    var rtc = window.rtc || window.PEX_RTC || null;
    if (!rtc) return false;
    function wrap(key){
      var prev = rtc[key];
      rtc[key] = function(){
        try { if (typeof prev === "function") prev.apply(this, arguments); }
        finally { try { refreshEndpointButtonsFromRoster(); } catch(e){} }
      };
    }
    ["onParticipantCreate","onParticipantUpdate","onParticipantDelete","onLayout","onDisconnect"].forEach(wrap);
    return true;
  }
  if (!bindRtc()){
    var t = setInterval(function(){ if (bindRtc()) clearInterval(t); }, 300);
  }

  document.addEventListener("DOMContentLoaded", function(){
    refreshEndpointButtonsFromRoster();
  });

  window.refreshEndpointButtonsFromRoster = refreshEndpointButtonsFromRoster;
})();

// ===== Final overlay (optional shims + presence + lock) =====
(function(){
  function stripSip(s){ s=String(s||"").trim().replace(/^sip:/i,""); return s.split("@")[0]; }
  function keyName(s){ return stripSip(s).toLowerCase().replace(/\s+/g,""); }
  function btn(){
    return document.getElementById("connectBtn")
        || document.getElementById("callButton")
        || document.querySelector("[data-role='callButton']")
        || document.querySelector(".callBtn");
  }
  function setUI(label){
    const b = btn(); if (!b) return;
    if (b.dataset) b.dataset.label = label;
    b.textContent = label;
    if (b.dataset) b.dataset.state = label.toLowerCase();
    b.disabled = false;
    if (/connected/i.test(label)) b.classList.add("green","connected");
    else b.classList.remove("green","connected");
  }
  function pm(){
    try { return (window.participantMap && typeof participantMap.forEach==="function") ? participantMap : null; }
    catch(e){ return null; }
  }
  function connectedTruth(){
    const m = pm();
    return !!(window.__rtcIsConnected || (m && m.size>0) || window.__watchedUuid);
  }

  ["resetCallButtonForName","resetAllCallButtons","reconcileActiveButtonsWithRoster"].forEach(function(fn){
    const prev = window[fn];
    if (typeof prev === "function"){
      window[fn] = function(){
        if (connectedTruth()) return;
        return prev.apply(this, arguments);
      };
    }
  });

  (function patchDial(){
    function setWatchTargetFromName(name){ window.__watchTarget = keyName(name||""); window.__watchedUuid = null; }
    const place = window.placeDialRequest;
    if (typeof place === "function"){
      window.placeDialRequest = function(name){
        try { setWatchTargetFromName(name); } catch(e){}
        return place.apply(this, arguments);
      };
    }
    const handle = window.handleCallButtonClick;
    if (typeof handle === "function"){
      window.handleCallButtonClick = function(btn, name){
        try { setWatchTargetFromName(name); } catch(e){}
        return handle.apply(this, arguments);
      };
    }
  })();

  function bindRtc(){
    const rtc = window.rtc || window.PEX_RTC || null;
    if (!rtc) return false;

    function wrap(key, after){
      const prev = rtc[key];
      rtc[key] = function(){
        try { if (typeof prev === "function") prev.apply(this, arguments); }
        finally { try { after && after.apply(this, arguments); } catch(e){} }
      };
    }

    wrap("onConnect", function(){ window.__rtcIsConnected = true; setUI("Connected to Current Session"); });

    wrap("onParticipantCreate", function(p){
      const disp = p && (p.display_name || p.remote_display_name || p.alias || "");
      if (window.__watchTarget && !window.__watchedUuid && keyName(disp) === window.__watchTarget){
        window.__watchedUuid = p.uuid;
      }
      if (connectedTruth()) setUI("Connected to Current Session");
    });

    wrap("onParticipantUpdate", function(p){
      if (window.__watchTarget && !window.__watchedUuid){
        const disp = p && (p.display_name || p.remote_display_name || p.alias || "");
        if (keyName(disp) === window.__watchTarget){
          window.__watchedUuid = p.uuid;
        }
      }
      if (connectedTruth()) setUI("Connected to Current Session");
    });

    wrap("onParticipantDelete", function(p){
      const m = pm();
      if (window.__watchedUuid && p && p.uuid === window.__watchedUuid){
        window.__watchedUuid = null; window.__watchTarget = null;
      }
      if (!connectedTruth() || (m && m.size===0)){
        window.__rtcIsConnected = false;
        setUI("Call");
      }
    });

    wrap("onLayout", function(){ if (connectedTruth()) setUI("Connected to Current Session"); });

    wrap("onDisconnect", function(){
      window.__rtcIsConnected = false;
      window.__watchedUuid = null; window.__watchTarget = null;
      setUI("Call");
    });

    if ("onCallDisconnected" in rtc){
      wrap("onCallDisconnected", function(){
        window.__rtcIsConnected = false;
        window.__watchedUuid = null; window.__watchTarget = null;
        setUI("Call");
      });
    }
    return true;
  }
  if (!bindRtc()){
    const t = setInterval(()=>{ if (bindRtc()) clearInterval(t); }, 300);
  }

  (function patchWatcher(){
    const prevStart = window.startParticipantWatcher;
    if (typeof prevStart !== "function") return;
    window.startParticipantWatcher = function(targetName, onPresent, onGone){
      const wrappedPresent = function(){ if (!connectedTruth()) try{ onPresent && onPresent(); }catch(e){} };
      const wrappedGone    = function(){ if (!connectedTruth()) try{ onGone && onGone(); }catch(e){} };
      return prevStart.call(this, targetName, wrappedPresent, wrappedGone);
    };
  })();

  setInterval(function(){
    if (!connectedTruth()) return;
    const b = btn(); if (!b) return;
    const label = (b.dataset && b.dataset.label) ? b.dataset.label : (b.textContent||"");
    if (String(label).toLowerCase().includes("call")) setUI("Connected to Current Session");
  }, 750);
})();

// ===== Per-endpoint pill sync (disconnect-safe) =====
(function () {
  function stripSip(s){ s=String(s||"").trim().replace(/^sip:/i,""); return s.split("@")[0]; }
  function key(s){ return stripSip(s).toLowerCase().replace(/\s+/g,""); }

  var ROOT_SEL = "#office-endpoints, .office-endpoints";

  function findEndpointRows(){
    var roots = document.querySelectorAll(ROOT_SEL + ", #cklab-office ul, #cklab-clinical ul, #vmrList");
    if (!roots || !roots.length) return [];
    var rows = [];
    roots.forEach(function(root){
      rows = rows.concat(Array.from(root.querySelectorAll("button")).map(function(btn){
        var explicit = btn.getAttribute("data-endpoint-key");
        if (explicit) return { btn, nameKey: explicit };
        var raw = btn.getAttribute("data-endpoint-name");
        if (!raw) {
          var row = btn.closest("div,li,tr") || btn.parentElement;
          var txt = "";
          if (row){
            for (var i=0;i<row.childNodes.length;i++){
              var n=row.childNodes[i];
              if (n===btn) break;
              if (n.nodeType===3) txt += n.textContent||"";
              if (n.nodeType===1 && !/button/i.test(n.tagName)) txt += " " + (n.textContent||"");
            }
          }
          raw = (txt||"").trim();
        }
        return { btn: btn, nameKey: key(raw||"") };
      }));
    });
    return rows.filter(function(r){ return !!r.nameKey; });
  }

  function rosterNameSet(){
    var set = new Set();
    try {
      if (window.participantMap && typeof participantMap.forEach === "function"){
        participantMap.forEach(function(p){
          var disp = p && (p.display_name || p.remote_display_name || p.alias || "");
          set.add(key(disp));
        });
      }
    } catch(e){}
    return set;
  }

  function setPill(btn, connected){
    var label = connected ? "Connected to Current Session" : "Call";
    if (btn.dataset) btn.dataset.label = label;
    btn.textContent = label;
    if (btn.dataset) btn.dataset.state = connected ? "connected" : "call";
    btn.classList.toggle("green", !!connected);
    btn.classList.toggle("connected", !!connected);
  }

  function syncEndpointPills(){
    var present = rosterNameSet();
    findEndpointRows().forEach(function(r){
      setPill(r.btn, present.has(r.nameKey));
    });
  }

  function clearAllPills(){
    findEndpointRows().forEach(function(r){ setPill(r.btn, false); });
  }

  function bindRtc(){
    var rtc = window.rtc || window.PEX_RTC || null;
    if (!rtc) return false;
    function wrap(k, after){
      var prev = rtc[k];
      rtc[k] = function(){
        try { if (typeof prev === "function") prev.apply(this, arguments); }
        finally { try { after && after.apply(this, arguments); } catch(e){} }
      };
    }
    ["onParticipantCreate","onParticipantUpdate","onParticipantDelete","onLayout"].forEach(function(k){
      wrap(k, syncEndpointPills);
    });
    wrap("onDisconnect", clearAllPills);
    if ("onCallDisconnected" in rtc) wrap("onCallDisconnected", clearAllPills);
    return true;
  }

  if (!bindRtc()){
    var t = setInterval(function(){ if (bindRtc()) clearInterval(t); }, 300);
  }

  document.addEventListener("DOMContentLoaded", function(){
    syncEndpointPills();
  });

  window.syncEndpointPills = syncEndpointPills;
})();

// ===== Poly Camera Switch (collision-safe, no toasts, console logs + inline status) =====
(function () {
  window.POLY_CFG = Object.assign({ baseUrl: "https://192.168.0.101" }, window.POLY_CFG || {});
  try { delete window.openCameraSwitch; } catch {}
  try { delete window.fetchPolyCameras; } catch {}
  try { delete window.submitCameraSwitch; } catch {}
  try { delete window.openCameraPresets; } catch {}

  function el(id) { return document.getElementById(id); }
  function setPolyStatus(msg){ const s = el("polyStatus"); if (s) s.textContent = msg; }

  // --- FECC visibility helpers (show/hide based on ptzcapable) ---
  function getFeccButtons() {
    return Array.from(document.querySelectorAll(
      "#openFeccBtn, #feccBtn, #feccToggle, [data-role='fecc'], .fecc-button, .feccBtn"
    ));
  }
  function setFeccVisibility(ptzCapable) {
    const els = getFeccButtons();
    els.forEach(el => {
      el.style.display = ptzCapable ? "" : "none";
      if (el.tagName === "BUTTON" || el.getAttribute("role")==="button") {
        el.disabled = !ptzCapable;
      }
    });
    window.__ptzCapableSelected = !!ptzCapable;
  }

  // Hardened boolean coercion
  const toBool = (v) => {
    if (typeof v === "boolean") return v;
    if (typeof v === "number")  return v !== 0;
    if (typeof v === "string") {
      const s = v.trim().toLowerCase();
      return ["true","yes","y","1","on","enabled"].includes(s);
    }
    return false;
  };

  // Extract a PTZ truthy flag from many possible shapes
  function extractPtzCapable(c){
    const direct =
      c.ptzcapable ?? c.ptzCapable ?? c["ptz_capable"] ?? c["ptz-capable"] ?? c.isPtzCapable ?? c["is-ptz-capable"];
    if (typeof direct !== "undefined") return toBool(direct);

    // Common nesting variants
    const cap = c.capabilities || c.features || c.attrs || {};
    const nested =
      cap.ptzcapable ?? cap.ptzCapable ?? cap["ptz-capable"] ?? cap["ptz_capable"] ??
      (cap.ptz && (cap.ptz.capable ?? cap.ptz.enabled ?? cap.ptz.supported ?? cap.ptz.available));
    if (typeof nested !== "undefined") return toBool(nested);

    const ptz = c.ptz || (cap && cap.ptz) || {};
    const ptzNested =
      ptz.capable ?? ptz.enabled ?? ptz.supported ?? ptz.available;
    if (typeof ptzNested !== "undefined") return toBool(ptzNested);

    // list style: capabilities: [{name:'ptz', enabled:true}]
    const list = c.capabilitiesList || cap.list || cap.items || [];
    if (Array.isArray(list)) {
      const hit = list.find(it => {
        const n = (it?.name || it?.id || "").toString().toLowerCase();
        return ["ptz","pan-tilt-zoom","fecc"].includes(n);
      });
      if (hit) return toBool(hit.enabled ?? hit.value ?? hit.supported);
    }

    return false; // default if unknown
  }

  window.openCameraSwitch = function () {
    const modal = el("cameraSwitchModal");
    if (!modal) { console.warn("cameraSwitchModal not found"); return; }
    setPolyStatus("Contacting device...");
    const listBlk = el("cameraListBlock");
    const goBtn   = el("switchCameraGo");
    if (listBlk) listBlk.style.display = "none";
    if (goBtn)   goBtn.style.display   = "none";
    modal.style.display = "flex";
    window.fetchPolyCameras();
  };

  window.closeCameraSwitchModal = function () {
    const modal = el("cameraSwitchModal");
    if (modal) modal.style.display = "none";
  };

  window.fetchPolyCameras = async function () {
    // use the outer el(), toBool(), extractPtzCapable()
    const select  = el("cameraSelect");
    const listBlk = el("cameraListBlock");
    const goBtn   = el("switchCameraGo");

    const isGeneric = (nm) =>
      /^\s*near\s*camera\s*$/i.test(nm || "") ||
      /^unknown$/i.test(nm || "") ||
      /^virtdev\d+$/i.test(nm || "") ||
      /^uilayer/i.test(nm || "") ||
      /^rdp/i.test(nm || "") ||
      /^ppcip$/i.test(nm || "") ||
      /^hdmi input$/i.test(nm || "");

    try {
      const res  = await fetch("/api/poly/cameras", { method: "POST" });
      const data = await res.json();
      if (!res.ok) throw new Error(data.error || res.statusText);

      const raw = Array.isArray(data) ? data : [];

      console.table(raw.map(c => ({
        cameraIndex: c.cameraIndex,
        name: c.name,
        selected: !!c.selected,
        nearCamera: c.nearCamera,
        connected_raw: c.connected,
        ptzcapable_raw: c.ptzcapable
      })));

      const normalized = raw.map((c) => {
        const name = (c.name || "").trim();

        const near = (typeof c.nearCamera === "undefined") ? true : toBool(c.nearCamera);
        const connected = (typeof c.connected !== "undefined")
          ? toBool(c.connected)
          : (near && !isGeneric(name));

        const direct    = (typeof c.ptzcapable !== "undefined") ? toBool(c.ptzcapable) : undefined;
        const hasPan    = ('hasPan'  in c) ? toBool(c.hasPan)  : undefined;
        const hasTilt   = ('hasTilt' in c) ? toBool(c.hasTilt) : undefined;
        const hasZoom   = ('hasZoom' in c) ? toBool(c.hasZoom) : undefined;
        const extracted = extractPtzCapable(c);

        // ✅ FIX: parenthesize the ?? part before using ||
        const ptzcapable = !!((direct ?? false) || hasPan || hasTilt || hasZoom || extracted);

        console.debug("[PTZ TRACE]", {
          idx: c.cameraIndex, name, direct, hasPan, hasTilt, hasZoom, extracted,
          decided_ptzcapable: ptzcapable
        });

        return {
          cameraIndex: c.cameraIndex,
          name,
          selected: !!c.selected,
          nearCamera: !!near,
          connected: !!connected,
          ptzcapable
        };
      });

      window.__polyCameras = normalized;

      console.log("[poly] normalized:");
      console.table(normalized);

      const items = normalized.filter((c) => c.nearCamera && (c.connected || c.selected) && !isGeneric(c.name));

      console.log("[poly] dropdown items:");
      console.table(items);

      const current =
        normalized.find(c => c.selected) ||
        normalized.find(c => c.nearCamera && c.connected) ||
        null;

      setFeccVisibility(current ? !!current.ptzcapable : false);

      if (!items.length) {
        setPolyStatus("No connected near cameras found.");
        if (select) select.innerHTML = "";
        if (listBlk) listBlk.style.display = "none";
        if (goBtn)  goBtn.style.display  = "none";
        setFeccVisibility(false);
        return;
      }

      if (select) {
        select.innerHTML = "";
        for (const c of items) {
          const opt = document.createElement("option");
          opt.value = String(c.cameraIndex);
          opt.textContent = c.name + (c.ptzcapable ? " • PTZ" : "");
          opt.dataset.ptz = String(!!c.ptzcapable);
          select.appendChild(opt);
        }
        select.onchange = function () {
          const idx = Number(this.value);
          const found = (window.__polyCameras || []).find(x => x.cameraIndex === idx);
          setFeccVisibility(!!(found && found.ptzcapable));
        };
      }

      setPolyStatus("Select a camera.");
      if (listBlk) listBlk.style.display = "block";
      if (goBtn)  goBtn.style.display  = "inline-block";

    } catch (err) {
      console.error("[Poly] fetch cameras failed:", err);
      setPolyStatus("Error: " + (err && err.message ? err.message : String(err)));
      // Hide FECC controls on error
      try {
        getFeccButtons().forEach(b => { b.style.display = "none"; b.disabled = true; });
      } catch {}
    }
  };

  window.submitCameraSwitch = async function () {
    const select = el("cameraSelect");
    const idx = select ? Number(select.value) : NaN;
    if (!idx && idx !== 0) { setPolyStatus("Choose a camera first."); return; }

    // Pre-toggle FECC based on current selection (no waiting for the re-fetch)
    try {
      const found = (window.__polyCameras && window.__polyCameras.find(x => x.cameraIndex === idx)) || null;
      if (found) setFeccVisibility(!!found.ptzcapable);
    } catch {}

    const payload = { camera_index: idx };

    // UI + console logging
    setPolyStatus(`Sending camera near ${idx}…`);
    console.groupCollapsed(`[Poly][SSH] switch -> ${idx}`);
    console.log("[Poly] request body:", payload);

    try {
      const res  = await fetch("/api/poly/switch", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(payload)
      });
      const data = await res.json();

      // Show server-side SSH timeline if present
      if (data && Array.isArray(data.debug)) {
        console.table(data.debug.map(d => ({
          ts: d.ts,
          event: d.event,
          user: d.user || "",
          host: d.host || "",
          cmd:  d.cmd || d.line || "",
          rc:   d.rc ?? "",
          note: d.variant || "",
          out_tail: d.out_tail || d.tail || ""
        })));
      }

      if (!res.ok) {
        console.error("[Poly] switch failed:", data && (data.error || data.message), data);
        throw new Error((data && data.error) || res.statusText);
      }

      console.log("[Poly] switch OK:", data);
      setPolyStatus(`Camera switched (index=${idx}).`);

      // Re-evaluate FECC visibility after switch (refresh selected+ptz from device)
      setTimeout(() => { try { window.fetchPolyCameras(); } catch(_) {} }, 400);
    } catch (err) {
      setPolyStatus("Switch failed: " + (err?.message || err));
      console.error("[Poly] switch failed:", err);
      setFeccVisibility(false);
    } finally {
      console.groupEnd();
    }
  };
})(); // <-- close the camera-switch IIFE

// === Camera Presets modal (no toasts; inline status + console) ===
(function () {
  function el(id) { return document.getElementById(id); }

  function ensurePresetsModal() {
    if (el("cameraPresetsModal")) return;
    const modal = document.createElement("div");
    modal.id = "cameraPresetsModal";
    modal.style.cssText = "position:fixed;inset:0;display:none;align-items:center;justify-content:center;background:rgba(0,0,0,0.45);z-index:9999;";
    modal.innerHTML = `
      <div style="background:#fff;width:min(980px,92vw);max-height:90vh;overflow:auto;border-radius:16px;padding:16px;box-shadow:0 10px 30px rgba(0,0,0,.2)">
        <div style="display:flex;align-items:center;justify-content:space-between;gap:8px;margin-bottom:8px">
          <h3 style="margin:0;font-size:18px">Near Camera Presets</h3>
          <button id="closePresetsBtn" class="button small">Close</button>
        </div>
        <div id="presetsStatus" style="font-size:13px;color:#666;margin:6px 0 10px;"></div>
        <div id="presetsGrid" style="display:grid;grid-template-columns:repeat(auto-fill,minmax(160px,1fr));gap:12px;"></div>
      </div>`;
    document.body.appendChild(modal);
    el("closePresetsBtn").onclick = () => { modal.style.display = "none"; };
  }

  async function loadPresets() {
    const status = el("presetsStatus");
    const grid = el("presetsGrid");
    if (status) status.textContent = "Loading presets…";
    if (grid) grid.innerHTML = "";

    try {
      const res = await fetch("/api/poly/presets", { method: "POST" });
      const data = await res.json();
      if (!res.ok) throw new Error((data && data.error) || res.statusText);

      if (!Array.isArray(data) || data.length === 0) {
        if (status) status.textContent = "No presets found.";
        return;
      }

      if (status) status.textContent = "Select a preset to activate.";
      for (const p of data) {
        const card = document.createElement("div");
        card.style.cssText = "border:1px solid #eee;border-radius:12px;padding:8px;display:flex;flex-direction:column;gap:6px;align-items:center;";

        const img = document.createElement("img");
        img.src = p.image;
        img.alt = "Preset " + p.index;
        img.style.cssText = "width:100%;aspect-ratio:4/3;object-fit:cover;border-radius:8px;background:#f6f6f6";
        card.appendChild(img);

        const row = document.createElement("div");
        row.style.cssText = "display:flex;align-items:center;justify-content:space-between;width:100%;gap:8px";
        const label = document.createElement("div");
        label.textContent = `Preset ${p.index}${p.stored ? "" : " (empty)"}`;
        label.style.cssText = "font-size:13px;color:#333";
        const btn = document.createElement("button");
        btn.className = "button small";
        btn.textContent = "Activate";
        btn.onclick = async () => {
          try {
            btn.disabled = true; btn.textContent = "Activating…";
            const r = await fetch("/api/poly/preset/activate", {
              method: "POST",
              headers: { "Content-Type": "application/json" },
              body: JSON.stringify({ preset_index: p.index })
            });
            const j = await r.json();
            if (!r.ok) throw new Error(j.error || r.statusText);
            console.log(`[Poly] preset ${p.index} activated.`);
            if (status) status.textContent = `Preset ${p.index} activated.`;
          } catch (e) {
            if (status) status.textContent = "Failed to activate preset: " + (e && e.message ? e.message : e);
            console.error("[Poly] preset activation failed:", e);
          } finally {
            btn.disabled = false; btn.textContent = "Activate";
          }
        };
        row.appendChild(label); row.appendChild(btn);
        card.appendChild(row);

        grid.appendChild(card);
      }
    } catch (err) {
      if (status) status.textContent = "Error: " + (err && err.message ? err.message : err);
      console.error("[Poly] load presets failed:", err);
    }
  }

  window.openCameraPresets = function () {
    ensurePresetsModal();
    const modal = el("cameraPresetsModal");
    if (modal) { modal.style.display = "flex"; }
    loadPresets();
  };
})(); // <-- close the presets IIFE

/* ===== Safety shim: ensure fetchAllEndpoints exists ===== */
(function () {
  async function _safe(fn, label){
    if (typeof fn === "function") return fn();
    console.warn(label + " not found on window");
    return Promise.resolve();
  }

  if (typeof window.fetchAllEndpoints !== "function") {
    window.fetchAllEndpoints = async function () {
      const syncBtn = document.getElementById("syncAllBtn");
      try {
        if (syncBtn) { syncBtn.disabled = true; syncBtn.textContent = "Syncing..."; }
        await Promise.all([
          _safe(window.fetchPexipEndpoints,       "fetchPexipEndpoints"),
          _safe(window.fetchPexipEndpointsDemo1,  "fetchPexipEndpointsDemo1"),
          _safe(window.fetchVMRs,                 "fetchVMRs")
        ]);
      } catch (e) {
        console.error("Global sync error (shim):", e);
        alert("Failed to sync: " + (e && e.message ? e.message : e));
      } finally {
        if (syncBtn) { syncBtn.disabled = false; syncBtn.textContent = "Sync endpoints from Pexip"; }
      }
    };
  }

  document.addEventListener("DOMContentLoaded", function () {
    const btn = document.getElementById("syncAllBtn");
    if (btn && !btn.__boundFetchAll) {
      btn.__boundFetchAll = true;
      btn.addEventListener("click", function (ev) {
        ev.preventDefault();
        window.fetchAllEndpoints();
      });
    }
  });
})();

// ===== FECC (Far End Camera Control) overlay: open/close/drag/wire =====
(function () {
  const panel = () => document.getElementById("feccPanel");

  // Make the panel draggable by its header
  function makeDraggable(root) {
    const header = root.querySelector(".fecc-header");
    if (!header) return;
    let startX = 0, startY = 0, origLeft = 0, origTop = 0, dragging = false;

    const onDown = (e) => {
      dragging = true;
      const evt = e.touches ? e.touches[0] : e;
      startX = evt.clientX; startY = evt.clientY;
      const rect = root.getBoundingClientRect();
      origLeft = rect.left; origTop = rect.top;
      document.addEventListener("mousemove", onMove);
      document.addEventListener("mouseup", onUp);
      document.addEventListener("touchmove", onMove, { passive: false });
      document.addEventListener("touchend", onUp);
    };
    const onMove = (e) => {
      if (!dragging) return;
      const evt = e.touches ? e.touches[0] : e;
      if (e.cancelable) e.preventDefault();
      const dx = evt.clientX - startX;
      const dy = evt.clientY - startY;
      root.style.left = Math.max(0, origLeft + dx) + "px";
      root.style.top  = Math.max(0, origTop  + dy) + "px";
    };
    const onUp = () => {
      dragging = false;
      document.removeEventListener("mousemove", onMove);
      document.removeEventListener("mouseup", onUp);
      document.removeEventListener("touchmove", onMove);
      document.removeEventListener("touchend", onUp);
    };

    header.addEventListener("mousedown", onDown);
    header.addEventListener("touchstart", onDown, { passive: true });
  }

  // Wire button clicks (no API calls yet)
  function wireButtons(root) {
    root.querySelectorAll(".fecc-btn").forEach(btn => {
      btn.addEventListener("click", () => {
        const action = btn.getAttribute("data-action");
        console.log("[FECC] action:", action);
        // TODO: call your FECC API here
      });
    });
    const presetBtn = root.querySelector("#feccPresetBtn");
    if (presetBtn) {
      presetBtn.addEventListener("click", () => {
        console.log("[FECC] set preset clicked");
        // TODO: call your preset-store API here
      });
    }
  }

  // Public open/close
  window.openFeccPanel = function () {
    const el = panel();
    if (!el) return console.warn("FECC panel element not found");
    if (!el.__wired) { makeDraggable(el); wireButtons(el); el.__wired = true; }
    el.style.display = "block";
  };
  window.closeFeccPanel = function () {
    const el = panel();
    if (el) el.style.display = "none";
  };
})();
